<?php
defined( 'ABSPATH' ) || exit;

/**
 * HRM IP Configuration Module
 * Provides admin-only access to configure and manage multiple IP addresses
 * for access control and attendance validation.
 */
class HRM_IP_Config {

    const OPTION_KEY = 'hrm_ip_whitelist';

    /**
     * Initialize the IP Configuration module
     */
    public static function init() {
        add_action( 'wp_ajax_hrm_add_ip', [ __CLASS__, 'ajax_add_ip' ] );
        add_action( 'wp_ajax_hrm_update_ip', [ __CLASS__, 'ajax_update_ip' ] );
        add_action( 'wp_ajax_hrm_delete_ip', [ __CLASS__, 'ajax_delete_ip' ] );
        add_action( 'wp_ajax_hrm_get_ips', [ __CLASS__, 'ajax_get_ips' ] );
        add_action( 'wp_ajax_hrm_toggle_ip_status', [ __CLASS__, 'ajax_toggle_ip_status' ] );
    }

    /**
     * Get all stored IP addresses
     * @return array
     */
    public static function get_all_ips() {
        $ips = get_option( self::OPTION_KEY, [] );
        return is_array( $ips ) ? $ips : [];
    }

    /**
     * Get only active/enabled IP addresses
     * @return array
     */
    public static function get_active_ips() {
        $all_ips = self::get_all_ips();
        return array_filter( $all_ips, function( $ip ) {
            return isset( $ip['enabled'] ) && $ip['enabled'] === true;
        });
    }

    /**
     * Validate IP address format (IPv4 and IPv6)
     * @param string $ip
     * @return bool
     */
    public static function validate_ip( $ip ) {
        return filter_var( $ip, FILTER_VALIDATE_IP ) !== false;
    }

    /**
     * Validate CIDR notation (e.g., 192.168.1.0/24)
     * @param string $cidr
     * @return bool
     */
    public static function validate_cidr( $cidr ) {
        $parts = explode( '/', $cidr );
        if ( count( $parts ) !== 2 ) {
            return false;
        }
        
        $ip = $parts[0];
        $mask = intval( $parts[1] );
        
        if ( ! self::validate_ip( $ip ) ) {
            return false;
        }
        
        // Check mask range based on IP version
        if ( filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 ) ) {
            return $mask >= 0 && $mask <= 32;
        } elseif ( filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6 ) ) {
            return $mask >= 0 && $mask <= 128;
        }
        
        return false;
    }

    /**
     * Validate IP or CIDR input
     * @param string $input
     * @return bool
     */
    public static function validate_ip_input( $input ) {
        $input = trim( $input );
        
        // Check if it's a CIDR notation
        if ( strpos( $input, '/' ) !== false ) {
            return self::validate_cidr( $input );
        }
        
        // Otherwise validate as regular IP
        return self::validate_ip( $input );
    }

    /**
     * Check if an IP is in the allowed list
     * @param string $ip_to_check
     * @return bool
     */
    public static function is_ip_allowed( $ip_to_check ) {
        $active_ips = self::get_active_ips();
        
        // If no IPs configured, allow all
        if ( empty( $active_ips ) ) {
            return true;
        }
        
        foreach ( $active_ips as $entry ) {
            $stored_ip = $entry['ip'];
            
            // Check CIDR notation
            if ( strpos( $stored_ip, '/' ) !== false ) {
                if ( self::ip_in_cidr( $ip_to_check, $stored_ip ) ) {
                    return true;
                }
            } else {
                // Exact match
                if ( $ip_to_check === $stored_ip ) {
                    return true;
                }
            }
        }
        
        return false;
    }

    /**
     * Check if IP is within a CIDR range
     * @param string $ip
     * @param string $cidr
     * @return bool
     */
    public static function ip_in_cidr( $ip, $cidr ) {
        list( $subnet, $mask ) = explode( '/', $cidr );
        
        if ( filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 ) ) {
            $ip_long = ip2long( $ip );
            $subnet_long = ip2long( $subnet );
            $mask_long = -1 << ( 32 - $mask );
            $subnet_long &= $mask_long;
            return ( $ip_long & $mask_long ) == $subnet_long;
        }
        
        // For IPv6, use a simpler approach
        return false; // IPv6 CIDR matching requires more complex logic
    }

    /**
     * Add a new IP address
     * @param string $ip
     * @param string $label
     * @param string $type
     * @return array Result with success status and message
     */
    public static function add_ip( $ip, $label = '', $type = 'office' ) {
        $ip = trim( sanitize_text_field( $ip ) );
        $label = sanitize_text_field( $label );
        $type = sanitize_key( $type );
        
        if ( empty( $ip ) ) {
            return [ 'success' => false, 'message' => 'IP address is required.' ];
        }
        
        if ( ! self::validate_ip_input( $ip ) ) {
            return [ 'success' => false, 'message' => 'Invalid IP address format.' ];
        }
        
        $ips = self::get_all_ips();
        
        // Check for duplicates
        foreach ( $ips as $existing ) {
            if ( $existing['ip'] === $ip ) {
                return [ 'success' => false, 'message' => 'This IP address already exists.' ];
            }
        }
        
        // Generate unique ID
        $id = uniqid( 'ip_' );
        
        $ips[] = [
            'id'         => $id,
            'ip'         => $ip,
            'label'      => $label ?: 'IP Address',
            'type'       => $type,
            'enabled'    => true,
            'created_at' => current_time( 'mysql' ),
            'created_by' => get_current_user_id()
        ];
        
        update_option( self::OPTION_KEY, $ips );
        
        // Log the action
        HRM_Security::log( 'system', "IP address added: {$ip} ({$label})" );
        
        return [ 'success' => true, 'message' => 'IP address added successfully.', 'id' => $id ];
    }

    /**
     * Update an existing IP address
     * @param string $id
     * @param array $data
     * @return array
     */
    public static function update_ip( $id, $data ) {
        $ips = self::get_all_ips();
        $updated = false;
        
        foreach ( $ips as $key => $entry ) {
            if ( $entry['id'] === $id ) {
                if ( isset( $data['ip'] ) ) {
                    $new_ip = trim( sanitize_text_field( $data['ip'] ) );
                    if ( ! self::validate_ip_input( $new_ip ) ) {
                        return [ 'success' => false, 'message' => 'Invalid IP address format.' ];
                    }
                    
                    // Check for duplicates (excluding current entry)
                    foreach ( $ips as $check_key => $check_entry ) {
                        if ( $check_key !== $key && $check_entry['ip'] === $new_ip ) {
                            return [ 'success' => false, 'message' => 'This IP address already exists.' ];
                        }
                    }
                    
                    $ips[$key]['ip'] = $new_ip;
                }
                
                if ( isset( $data['label'] ) ) {
                    $ips[$key]['label'] = sanitize_text_field( $data['label'] );
                }
                
                if ( isset( $data['type'] ) ) {
                    $ips[$key]['type'] = sanitize_key( $data['type'] );
                }
                
                if ( isset( $data['enabled'] ) ) {
                    $ips[$key]['enabled'] = (bool) $data['enabled'];
                }
                
                $ips[$key]['updated_at'] = current_time( 'mysql' );
                $ips[$key]['updated_by'] = get_current_user_id();
                
                $updated = true;
                break;
            }
        }
        
        if ( ! $updated ) {
            return [ 'success' => false, 'message' => 'IP address not found.' ];
        }
        
        update_option( self::OPTION_KEY, $ips );
        
        HRM_Security::log( 'system', "IP address updated: ID {$id}" );
        
        return [ 'success' => true, 'message' => 'IP address updated successfully.' ];
    }

    /**
     * Delete an IP address
     * @param string $id
     * @return array
     */
    public static function delete_ip( $id ) {
        $ips = self::get_all_ips();
        $deleted_ip = '';
        
        foreach ( $ips as $key => $entry ) {
            if ( $entry['id'] === $id ) {
                $deleted_ip = $entry['ip'];
                unset( $ips[$key] );
                break;
            }
        }
        
        if ( empty( $deleted_ip ) ) {
            return [ 'success' => false, 'message' => 'IP address not found.' ];
        }
        
        // Re-index array
        $ips = array_values( $ips );
        
        update_option( self::OPTION_KEY, $ips );
        
        HRM_Security::log( 'system', "IP address deleted: {$deleted_ip}" );
        
        return [ 'success' => true, 'message' => 'IP address deleted successfully.' ];
    }

    /**
     * AJAX handler: Add IP
     */
    public static function ajax_add_ip() {
        check_ajax_referer( 'hrm_ajax_nonce', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Unauthorized access.' ] );
        }
        
        $ip = isset( $_POST['ip'] ) ? $_POST['ip'] : '';
        $label = isset( $_POST['label'] ) ? $_POST['label'] : '';
        $type = isset( $_POST['type'] ) ? $_POST['type'] : 'office';
        
        $result = self::add_ip( $ip, $label, $type );
        
        if ( $result['success'] ) {
            wp_send_json_success( $result );
        } else {
            wp_send_json_error( $result );
        }
    }

    /**
     * AJAX handler: Update IP
     */
    public static function ajax_update_ip() {
        check_ajax_referer( 'hrm_ajax_nonce', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Unauthorized access.' ] );
        }
        
        $id = isset( $_POST['id'] ) ? sanitize_text_field( $_POST['id'] ) : '';
        $data = [
            'ip'    => isset( $_POST['ip'] ) ? $_POST['ip'] : null,
            'label' => isset( $_POST['label'] ) ? $_POST['label'] : null,
            'type'  => isset( $_POST['type'] ) ? $_POST['type'] : null
        ];
        
        // Remove null values
        $data = array_filter( $data, function( $v ) { return $v !== null; } );
        
        $result = self::update_ip( $id, $data );
        
        if ( $result['success'] ) {
            wp_send_json_success( $result );
        } else {
            wp_send_json_error( $result );
        }
    }

    /**
     * AJAX handler: Delete IP
     */
    public static function ajax_delete_ip() {
        check_ajax_referer( 'hrm_ajax_nonce', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Unauthorized access.' ] );
        }
        
        $id = isset( $_POST['id'] ) ? sanitize_text_field( $_POST['id'] ) : '';
        
        $result = self::delete_ip( $id );
        
        if ( $result['success'] ) {
            wp_send_json_success( $result );
        } else {
            wp_send_json_error( $result );
        }
    }

    /**
     * AJAX handler: Get all IPs
     */
    public static function ajax_get_ips() {
        check_ajax_referer( 'hrm_ajax_nonce', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Unauthorized access.' ] );
        }
        
        $ips = self::get_all_ips();
        wp_send_json_success( [ 'ips' => $ips ] );
    }

    /**
     * AJAX handler: Toggle IP status
     */
    public static function ajax_toggle_ip_status() {
        check_ajax_referer( 'hrm_ajax_nonce', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Unauthorized access.' ] );
        }
        
        $id = isset( $_POST['id'] ) ? sanitize_text_field( $_POST['id'] ) : '';
        $enabled = isset( $_POST['enabled'] ) ? ( $_POST['enabled'] === 'true' ) : false;
        
        $result = self::update_ip( $id, [ 'enabled' => $enabled ] );
        
        if ( $result['success'] ) {
            wp_send_json_success( $result );
        } else {
            wp_send_json_error( $result );
        }
    }

    /**
     * Render the IP Configuration tab content
     */
    public static function render_ip_config_tab() {
        $ips = self::get_all_ips();
        $current_ip = isset( $_SERVER['REMOTE_ADDR'] ) ? $_SERVER['REMOTE_ADDR'] : 'Unknown';
        $active_count = count( array_filter( $ips, function( $ip ) { return $ip['enabled']; } ) );
        ?>
        <div class="hrm-ip-config-wrapper">
            <!-- Page Header -->
            <div class="hrm-ip-page-header">
                <div class="hrm-ip-header-content">
                    <div class="hrm-ip-header-icon">
                        <span class="dashicons dashicons-shield-alt"></span>
                    </div>
                    <div class="hrm-ip-header-text">
                        <h2>IP Address Configuration</h2>
                        <p>Control which IP addresses are allowed for employee attendance check-in. Configure office networks, VPNs, and remote worker locations.</p>
                    </div>
                </div>
                <div class="hrm-ip-header-stats">
                    <div class="hrm-ip-stat-box">
                        <span class="stat-number" id="active-ip-count"><?php echo $active_count; ?></span>
                        <span class="stat-label">Active IPs</span>
                    </div>
                    <div class="hrm-ip-stat-box">
                        <span class="stat-number" id="total-ip-count"><?php echo count( $ips ); ?></span>
                        <span class="stat-label">Total IPs</span>
                    </div>
                </div>
            </div>

            <!-- Current IP Banner -->
            <div class="hrm-ip-current-banner">
                <div class="hrm-ip-banner-left">
                    <span class="hrm-ip-banner-icon">
                        <span class="dashicons dashicons-location"></span>
                    </span>
                    <div class="hrm-ip-banner-text">
                        <span class="hrm-ip-banner-label">Your Current IP Address</span>
                        <code id="current-ip-display"><?php echo esc_html( $current_ip ); ?></code>
                    </div>
                </div>
                <button type="button" class="hrm-btn-primary btn-sm" id="add-current-ip-btn">
                    <span class="dashicons dashicons-plus-alt2"></span>
                    Quick Add My IP
                </button>
            </div>

            <!-- Main Content Grid -->
            <div class="hrm-ip-main-grid">
                <!-- Left Column: Add New IP -->
                <div class="hrm-ip-add-section">
                    <div class="hrm-ip-card">
                        <div class="hrm-ip-card-header">
                            <span class="dashicons dashicons-plus-alt"></span>
                            <h3>Add New IP Address</h3>
                        </div>
                        <div class="hrm-ip-card-body">
                            <div class="hrm-ip-form">
                                <div class="hrm-ip-field">
                                    <label for="new-ip-address">
                                        IP Address <span class="required">*</span>
                                    </label>
                                    <div class="hrm-ip-input-wrapper">
                                        <span class="hrm-ip-input-icon">
                                            <span class="dashicons dashicons-admin-network"></span>
                                        </span>
                                        <input type="text" id="new-ip-address" 
                                               placeholder="e.g., 192.168.1.1 or 192.168.1.0/24" 
                                               class="hrm-ip-input">
                                    </div>
                                    <span class="hrm-ip-field-hint">
                                        <span class="dashicons dashicons-info-outline"></span>
                                        Supports IPv4, IPv6, and CIDR notation
                                    </span>
                                </div>

                                <div class="hrm-ip-field">
                                    <label for="new-ip-label">Label / Description</label>
                                    <div class="hrm-ip-input-wrapper">
                                        <span class="hrm-ip-input-icon">
                                            <span class="dashicons dashicons-tag"></span>
                                        </span>
                                        <input type="text" id="new-ip-label" 
                                               placeholder="e.g., Main Office, Remote Worker" 
                                               class="hrm-ip-input">
                                    </div>
                                </div>

                                <div class="hrm-ip-field">
                                    <label for="new-ip-type">Location Type</label>
                                    <div class="hrm-ip-select-wrapper">
                                        <select id="new-ip-type" class="hrm-ip-select">
                                            <option value="office">🏢 Office</option>
                                            <option value="branch">🏬 Branch Office</option>
                                            <option value="remote">💻 Remote / WFH</option>
                                            <option value="vpn">🔒 VPN</option>
                                            <option value="other">📍 Other</option>
                                        </select>
                                        <span class="hrm-ip-select-arrow">
                                            <span class="dashicons dashicons-arrow-down-alt2"></span>
                                        </span>
                                    </div>
                                </div>

                                <button type="button" id="add-ip-btn" class="hrm-btn-primary hrm-ip-submit-btn">
                                    <span class="dashicons dashicons-plus-alt2"></span>
                                    Add IP Address
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Info Box -->
                    <div class="hrm-ip-info-card">
                        <div class="hrm-ip-info-icon">
                            <span class="dashicons dashicons-lightbulb"></span>
                        </div>
                        <div class="hrm-ip-info-content">
                            <h4>How It Works</h4>
                            <ul>
                                <li>Only employees connecting from allowed IPs can mark attendance</li>
                                <li>If no IPs are configured, check-in is allowed from anywhere</li>
                                <li>Toggle IPs on/off without deleting them</li>
                                <li>Use CIDR notation (e.g., 192.168.1.0/24) for IP ranges</li>
                            </ul>
                        </div>
                    </div>
                </div>

                <!-- Right Column: IP List -->
                <div class="hrm-ip-list-section">
                    <div class="hrm-ip-card hrm-ip-list-card">
                        <div class="hrm-ip-card-header">
                            <span class="dashicons dashicons-list-view"></span>
                            <h3>Configured IP Addresses</h3>
                        </div>
                        <div class="hrm-ip-card-body no-padding">
                            <div class="hrm-ip-list-container" id="ip-list-container">
                                <?php if ( empty( $ips ) ) : ?>
                                    <div class="hrm-ip-empty-state" id="ip-empty-state">
                                        <div class="hrm-ip-empty-icon">
                                            <span class="dashicons dashicons-shield"></span>
                                        </div>
                                        <h4>No IP Addresses Configured</h4>
                                        <p>Add IP addresses to restrict attendance check-in to specific locations.</p>
                                        <span class="hrm-ip-empty-hint">
                                            <span class="dashicons dashicons-info"></span>
                                            If no IPs are configured, employees can check in from anywhere.
                                        </span>
                                    </div>
                                <?php else : ?>
                                    <?php foreach ( $ips as $ip_entry ) : ?>
                                        <div class="hrm-ip-item <?php echo $ip_entry['enabled'] ? 'enabled' : 'disabled'; ?>" data-id="<?php echo esc_attr( $ip_entry['id'] ); ?>">
                                            <div class="hrm-ip-item-toggle">
                                                <label class="hrm-toggle">
                                                    <input type="checkbox" class="ip-toggle-checkbox" <?php checked( $ip_entry['enabled'], true ); ?>>
                                                    <span class="hrm-toggle-slider"></span>
                                                </label>
                                            </div>
                                            <div class="hrm-ip-item-icon <?php echo esc_attr( $ip_entry['type'] ); ?>">
                                                <span class="dashicons <?php echo self::get_type_icon( $ip_entry['type'] ); ?>"></span>
                                            </div>
                                            <div class="hrm-ip-item-content">
                                                <div class="hrm-ip-item-header">
                                                    <span class="hrm-ip-label"><?php echo esc_html( $ip_entry['label'] ); ?></span>
                                                    <span class="hrm-ip-type-badge <?php echo esc_attr( $ip_entry['type'] ); ?>"><?php echo esc_html( ucfirst( $ip_entry['type'] ) ); ?></span>
                                                    <?php if ( $ip_entry['ip'] === $current_ip ) : ?>
                                                        <span class="hrm-ip-you-badge">You</span>
                                                    <?php endif; ?>
                                                </div>
                                                <div class="hrm-ip-item-address">
                                                    <code><?php echo esc_html( $ip_entry['ip'] ); ?></code>
                                                </div>
                                            </div>
                                            <div class="hrm-ip-item-actions">
                                                <button type="button" class="hrm-ip-action-btn edit-ip-btn" title="Edit">
                                                    <span class="dashicons dashicons-edit"></span>
                                                </button>
                                                <button type="button" class="hrm-ip-action-btn delete-ip-btn" title="Delete">
                                                    <span class="dashicons dashicons-trash"></span>
                                                </button>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Edit IP Modal -->
        <div class="hrm-modal" id="edit-ip-modal" style="display: none;">
            <div class="hrm-modal-content">
                <div class="hrm-modal-header">
                    <h3><span class="dashicons dashicons-edit"></span> Edit IP Address</h3>
                    <button type="button" class="hrm-close-modal">&times;</button>
                </div>
                <div class="hrm-modal-body">
                    <input type="hidden" id="edit-ip-id">
                    <div class="hrm-form-group">
                        <label for="edit-ip-address">IP Address <span class="required">*</span></label>
                        <input type="text" id="edit-ip-address" class="hrm-input">
                    </div>
                    <div class="hrm-form-group">
                        <label for="edit-ip-label">Label</label>
                        <input type="text" id="edit-ip-label" class="hrm-input">
                    </div>
                    <div class="hrm-form-group">
                        <label for="edit-ip-type">Type</label>
                        <select id="edit-ip-type" class="hrm-select">
                            <option value="office">Office</option>
                            <option value="branch">Branch Office</option>
                            <option value="remote">Remote/WFH</option>
                            <option value="vpn">VPN</option>
                            <option value="other">Other</option>
                        </select>
                    </div>
                </div>
                <div class="hrm-modal-footer">
                    <button type="button" class="hrm-btn-outline cancel-modal-btn">Cancel</button>
                    <button type="button" class="hrm-btn-primary" id="save-ip-btn">
                        <span class="dashicons dashicons-saved"></span> Save Changes
                    </button>
                </div>
            </div>
        </div>

        <!-- Delete Confirmation Modal -->
        <div class="hrm-modal" id="delete-ip-modal" style="display: none;">
            <div class="hrm-modal-content hrm-modal-sm">
                <div class="hrm-modal-header">
                    <h3><span class="dashicons dashicons-warning" style="color: #EF4444;"></span> Confirm Deletion</h3>
                    <button type="button" class="hrm-close-modal">&times;</button>
                </div>
                <div class="hrm-modal-body">
                    <input type="hidden" id="delete-ip-id">
                    <p>Are you sure you want to delete this IP address?</p>
                    <p class="hrm-text-muted"><code id="delete-ip-display"></code></p>
                    <p class="hrm-text-danger">This action cannot be undone.</p>
                </div>
                <div class="hrm-modal-footer">
                    <button type="button" class="hrm-btn-outline cancel-modal-btn">Cancel</button>
                    <button type="button" class="hrm-btn-red" id="confirm-delete-ip-btn">
                        <span class="dashicons dashicons-trash"></span> Delete IP
                    </button>
                </div>
            </div>
        </div>

        <script>
        jQuery(document).ready(function($) {
            const nonce = hrm_admin_params.nonce;
            const ajaxurl = hrm_admin_params.ajaxurl;

            function updateStats() {
                const total = $('.hrm-ip-item').length;
                const active = $('.hrm-ip-item.enabled').length;
                $('#total-ip-count').text(total);
                $('#active-ip-count').text(active);
                
                if (total === 0) {
                    $('#ip-empty-state').show();
                } else {
                    $('#ip-empty-state').hide();
                }
            }

            function showToast(message, type = 'success') {
                const toastHtml = `
                    <div class="hrm-toast-notification ${type}">
                        <span class="dashicons dashicons-${type === 'success' ? 'yes-alt' : 'warning'}"></span>
                        <span>${message}</span>
                        <button class="hrm-toast-close" onclick="this.parentElement.remove()">&times;</button>
                    </div>
                `;
                $('.hrm-ip-config-card').prepend(toastHtml);
                setTimeout(function() {
                    $('.hrm-toast-notification').first().fadeOut(300, function() { $(this).remove(); });
                }, 4000);
            }

            function getTypeIcon(type) {
                const icons = {
                    'office': 'dashicons-building',
                    'branch': 'dashicons-store',
                    'remote': 'dashicons-laptop',
                    'vpn': 'dashicons-shield',
                    'other': 'dashicons-admin-network'
                };
                return icons[type] || 'dashicons-admin-network';
            }

            function createIpItemHtml(ip) {
                const currentIp = $('#current-ip-display').text();
                const isCurrentIp = ip.ip === currentIp;
                const enabledClass = ip.enabled ? 'enabled' : 'disabled';
                const checkedAttr = ip.enabled ? 'checked' : '';
                const typeLabel = ip.type.charAt(0).toUpperCase() + ip.type.slice(1);
                
                return `
                    <div class="hrm-ip-item ${enabledClass}" data-id="${ip.id}">
                        <div class="hrm-ip-item-toggle">
                            <label class="hrm-toggle">
                                <input type="checkbox" class="ip-toggle-checkbox" ${checkedAttr}>
                                <span class="hrm-toggle-slider"></span>
                            </label>
                        </div>
                        <div class="hrm-ip-item-icon ${ip.type}">
                            <span class="dashicons ${getTypeIcon(ip.type)}"></span>
                        </div>
                        <div class="hrm-ip-item-content">
                            <div class="hrm-ip-item-header">
                                <span class="hrm-ip-label">${ip.label || 'IP Address'}</span>
                                <span class="hrm-ip-type-badge ${ip.type}">${typeLabel}</span>
                                ${isCurrentIp ? '<span class="hrm-ip-you-badge">You</span>' : ''}
                            </div>
                            <div class="hrm-ip-item-address">
                                <code>${ip.ip}</code>
                            </div>
                        </div>
                        <div class="hrm-ip-item-actions">
                            <button type="button" class="hrm-ip-action-btn edit-ip-btn" title="Edit">
                                <span class="dashicons dashicons-edit"></span>
                            </button>
                            <button type="button" class="hrm-ip-action-btn delete-ip-btn" title="Delete">
                                <span class="dashicons dashicons-trash"></span>
                            </button>
                        </div>
                    </div>
                `;
            }


            // Add current IP
            $('#add-current-ip-btn').on('click', function() {
                const currentIp = $('#current-ip-display').text();
                $('#new-ip-address').val(currentIp);
                $('#new-ip-label').val('My Current Location').focus();
            });


            // Edit IP - Open Modal
            $(document).on('click', '.edit-ip-btn', function() {
                const item = $(this).closest('.hrm-ip-item');
                const id = item.data('id');
                const ip = item.find('.hrm-ip-item-address code').text();
                const label = item.find('.hrm-ip-label').text();
                const type = item.find('.hrm-ip-type-badge').attr('class').split(' ').pop();

                $('#edit-ip-id').val(id);
                $('#edit-ip-address').val(ip);
                $('#edit-ip-label').val(label);
                $('#edit-ip-type').val(type);

                $('#edit-ip-modal').fadeIn(200);
            });


            // Delete IP - Open Modal
            $(document).on('click', '.delete-ip-btn', function() {
                const item = $(this).closest('.hrm-ip-item');
                const id = item.data('id');
                const ip = item.find('.hrm-ip-item-address code').text();

                $('#delete-ip-id').val(id);
                $('#delete-ip-display').text(ip);

                $('#delete-ip-modal').fadeIn(200);
            });


            // Close modals
            $('.hrm-close-modal, .cancel-modal-btn').on('click', function() {
                $(this).closest('.hrm-modal').fadeOut(200);
            });

            $(window).on('click', function(e) {
                if ($(e.target).hasClass('hrm-modal')) {
                    $('.hrm-modal').fadeOut(200);
                }
            });

            // Initial stats update
            updateStats();
        });
        </script>
        <?php
    }

    /**
     * Get icon based on IP type
     * @param string $type
     * @return string
     */
    public static function get_type_icon( $type ) {
        $icons = [
            'office' => 'dashicons-building',
            'branch' => 'dashicons-store',
            'remote' => 'dashicons-laptop',
            'vpn'    => 'dashicons-shield',
            'other'  => 'dashicons-admin-network'
        ];
        return isset( $icons[$type] ) ? $icons[$type] : 'dashicons-admin-network';
    }
}
